# projects/python_generator/processing/segmenter.py

import pandas as pd
from .formatters import get_raw_value

def determine_company_segment(company_main_data: pd.Series, financial_data: dict) -> dict:
    """
    Nosaka uzņēmuma segmentu, balstoties uz tā statusu, juridisko formu un finanšu datiem.
    Atgriež vārdnīcu ar segmenta informāciju.
    """
    segment = {
        'status': 'Nezināms',  # Var būt: Aktīvs, Likvidēts, Reorganizēts, Cits
        'form_group': 'Cits',  # Var būt: Komercsabiedrība, NVO, Partija, Cits
        'financials': 'Nav datu' # Var būt: Peļņa, Zaudējumi, Nav datu
    }

    if company_main_data is None or company_main_data.empty:
        return segment

    # 1. Nosakām statusa segmentu
    closed_val = get_raw_value(company_main_data, 'closed')
    terminated_date = get_raw_value(company_main_data, 'terminated')

    if closed_val == 'L':
        segment['status'] = 'Likvidēts'
    elif closed_val == 'R':
        segment['status'] = 'Reorganizēts'
    elif terminated_date and str(terminated_date).strip() and str(terminated_date) != '0000-00-00':
        segment['status'] = 'Cits' # Darbība izbeigta
    else:
        segment['status'] = 'Aktīvs'

    # 2. Nosakām juridiskās formas segmentu
    type_text = get_raw_value(company_main_data, 'type_text')
    if type_text:
        type_text_lower = type_text.lower()
        if any(s in type_text_lower for s in ['sabiedrība', 'komersants', 'saimniecība', 'uzņēmums']):
            segment['form_group'] = 'Komercsabiedrība'
        elif any(s in type_text_lower for s in ['biedrība', 'nodibinājums']):
            segment['form_group'] = 'NVO'
        elif 'partija' in type_text_lower:
            segment['form_group'] = 'Partija'

    # 3. Nosakām finanšu segmentu (tikai aktīviem komercuzņēmumiem)
    if segment['status'] == 'Aktīvs' and segment['form_group'] == 'Komercsabiedrība':
        summary_data_ugp = financial_data.get('UGP', [])
        if summary_data_ugp:
            latest_report = summary_data_ugp[0] # Pieņemam, ka saraksts ir sakārtots no jaunākā
            profit = latest_report.get('profit')
            if profit is not None:
                segment['financials'] = 'Peļņa' if profit > 0 else 'Zaudējumi'

    return segment