# projects/python_generator/processing/formatters.py

import pandas as pd
import re
import math
from datetime import datetime

def sanitize_for_json(obj):
    """
    Rekursīvi pārbauda objektu (vārdnīcu vai sarakstu) un aizstāj
    ne-JSON saderīgas vērtības (kā NaN, inf, pd.NA) ar None.
    """
    if isinstance(obj, dict):
        return {k: sanitize_for_json(v) for k, v in obj.items()}
    elif isinstance(obj, list):
        return [sanitize_for_json(elem) for elem in obj]
    # Pārbaude uz float nan/inf un pandas NA/NaT
    elif pd.isna(obj) or (isinstance(obj, float) and not math.isfinite(obj)):
        return None
    return obj

def get_raw_value(data_row, key):
    """
    Iegūst "tīro" vērtību no pandas Series (rindas) vai vārdnīcas.
    Apstrādā None, NaN un tukšas virknes.
    """
    if data_row is None:
        return None

    value = None
    if isinstance(data_row, pd.Series):
        if key in data_row.index:
            value = data_row[key]
        else:
            return None
    elif isinstance(data_row, dict):
        value = data_row.get(key)
    else:
        try:
            value = getattr(data_row, key, None)
        except AttributeError:
            value = None

    if pd.isna(value):
        return None

    s_value = str(value).strip()
    if s_value == '':
        return None

    if key in ['registered', 'terminated', 'reregistration_term']:
        if s_value == '0000-00-00' or s_value.startswith('0000-00-00'):
            return None
    return value

def format_raw_date(data_row, key):
    """
    Formatē datumu no<y_bin_46>-MM-DD HH:MM:SS uz<x_bin_398>-MM-DD.
    """
    value = get_raw_value(data_row, key)
    if value is not None:
        s_value = str(value)
        try:
            if isinstance(value, (datetime, pd.Timestamp)):
                return value.strftime('%Y-%m-%d')
            dt_obj = pd.to_datetime(s_value, errors='coerce')
            if pd.isna(dt_obj):
                if re.match(r'^\d{4}-\d{2}-\d{2}', s_value):
                    return s_value[:10]
                return s_value
            return dt_obj.strftime('%Y-%m-%d')
        except Exception:
            if re.match(r'^\d{4}-\d{2}-\d{2}', s_value):
                return s_value[:10]
            return s_value
    return None

def prepare_for_chart(value):
    """
    Sagatavo skaitlisku vērtību diagrammai, apstrādājot NaN, bezgalību un
    kļūdainus formātus.
    """
    if pd.isna(value):
        return None
    s_value = str(value).strip()
    if s_value == '':
        return None

    try:
        s_value_cleaned = s_value.replace(',', '.')
        num = float(s_value_cleaned)
        if math.isnan(num) or math.isinf(num):
            return None
        return num
    except (ValueError, TypeError):
        return None

def format_number_data(value, currency=' ', rounding_factor=1):
    """
    Formatē skaitli ar atdalītājiem un valūtu.
    """
    num = prepare_for_chart(value)
    if num is None:
        return '—'
    actual_value = num * rounding_factor
    if actual_value == int(actual_value):
        formatted_num = f"{int(actual_value):,}".replace(",", " ")
    else:
        formatted_num = f"{actual_value:,.0f}".replace(",", " ")

    return f"{formatted_num} {currency.strip() if currency else ''}"


def normalize_name(name):
    """
    Normalizē uzņēmuma nosaukumu meklēšanas vajadzībām.
    """
    if name is None or pd.isna(name):
        return ''
    normalized = str(name).lower()
    chars_to_space = ['-', '&', '@', '%', '+', '=']
    for char in chars_to_space:
        normalized = normalized.replace(char, ' ')

    chars_to_remove = ['.', ',', ':', ';', '!', '?', '(', ')', '[', ']', '{', '}', '"', "'", '«', '»']
    for char in chars_to_remove:
        normalized = normalized.replace(char, '')

    normalized = re.sub(r'\s+', ' ', normalized).strip()
    return normalized